#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include <sys/ioctl.h>

#include "liblif.h"

int usage(void);
void print_lifdir(struct lifdir *ld);

extern char *__progname;

static u_int8_t buf[SECTOR_SIZE];
static int Lflag, iflag, lflag, vflag;
static int termwidth = 80;		/* default terminal width */
static int singlecol = 0;		/* use single column output */

int
main(int argc, char *argv[])
{
	int fd, ch, i;
	struct tm tm;
	struct lifvol *lv;
	struct lifdir *ld;
	const char *p;
	struct winsize win;
	char volname[7];
	int blocknum;
	
	if (isatty(STDOUT_FILENO)) {
		if ((p = getenv("COLUMNS")) != NULL)
			termwidth = atoi(p);
		else if (ioctl(STDOUT_FILENO, TIOCGWINSZ, &win) == 0 &&
		    win.ws_col > 0)
			termwidth = win.ws_col;
	} else
		singlecol = 1;

	Lflag = iflag = lflag = vflag = 0;
	while ((ch = getopt(argc, argv, "ilvCL")) != -1) {
		switch (ch) {
		case 'i':
			iflag = 1;
			break;
		case 'l':
			lflag = 1;
			break;
		case 'v':
			vflag = 1;
			break;
		case 'C':
			singlecol = 0;
			break;
		case 'L':
			Lflag = 1;
			break;
		case '?':
		default:
			usage();
			/* NOTREACHED */
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 1)
		usage();

	fd = open(argv[0], O_RDONLY,0);
	if (fd < 0)
		err(1, "failed to open %s", argv[0]);

	/* read block 0 */
	lif_block_read(fd, 0, buf);

	lv = (struct lifvol *)buf;
	if (be16toh(lv->lv_magic) != VOL_MAGIC)
		err(1, "not a LIF volume");

	if (vflag) {
		if (lv->lv_label[0] != ' ') {
			for (i=0; i<6; i++)
				volname[i] = lv->lv_label[i];
			volname[i] = '\0';
		} else
			strncpy(volname, "(none)", 6);

		printf("Version: %d\n", be16toh(lv->lv_version));
		printf("Volume: %s, %d bytes", volname,
		    SECTOR_SIZE * be16toh(lv->lv_length));
		if (lv->lv_toc.toc_month != 0) {
			lif_gettime(&lv->lv_toc, &tm);
			printf(", %s", asctime(&tm));
		} else
			printf("\n");
		printf("Directory entries: %d\n", be32toh(lv->lv_dirsize));
	}

	blocknum = htobe32(lv->lv_dirstart);
	for (i=0; i<htobe32(lv->lv_dirsize); i++) {

		if ((i % (SECTOR_SIZE/sizeof(struct lifdir))) == 0) {
			lif_block_read(fd, blocknum++, buf);
			ld = (struct lifdir *)buf;
		}

		if (be16toh(ld->ld_type) == 0)
			continue;	/* deleted file */
		if (be16toh(ld->ld_type) == 0xffff)
			goto end;	/* directory end */
		print_lifdir(ld);
		ld++;
	}

end:
	close(fd);

	exit(0);
	/* NOTREACHED */
}


int
usage(void)
{

	fprintf(stderr, "Usage: %s [-CilLv] name\n", __progname);
	exit(EXIT_FAILURE);
}


void
print_lifdir(struct lifdir *ld)
{
	struct tm tm;
	int i;

	for (i=0; i<10; i++)
		putchar(ld->ld_name[i]);

	if (lflag) {
		printf("\t%5d\t%5d", htobe16(ld->ld_type),
		    htobe32(ld->ld_length));

		if (ld->ld_toc.toc_month != 0) {
			lif_gettime(&ld->ld_toc, &tm);
			printf(" %s", asctime(&tm));
		} else
			printf("\n");
	} else
		printf("\n");
}
